<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Profile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Spatie\Permission\Models\Role;
use App\Models\Department;

class AuthController extends Controller
{
    /**
     * Redirect path after login/registration
     */
    protected $redirectTo = '/dashboard';

    /**
     * Username field for authentication
     */
    protected $username = 'username';

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->middleware('guest')->except('logout');
    }

    /**
     * Show login form
     */
    public function showLoginForm()
    {
        return view('auth.login');
    }

    /**
     * Handle login request
     */
    public function login(Request $request)
    {
        $request->validate([
            'username' => 'required|string',
            'password' => 'required|string',
        ]);

        $credentials = $request->only('username', 'password');

        if (Auth::attempt($credentials, $request->filled('remember'))) {
            $request->session()->regenerate();
            return redirect()->intended($this->redirectTo);
        }

        return back()->withErrors(['username' => 'Invalid credentials'])
                     ->withInput($request->only('username'));
    }

    /**
     * Logout user
     */
    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect('/login');
    }

    /**
     * Show registration form
     */
    public function showRegistrationForm()
    {
        $departments = Department::pluck('department_name', 'id');
        $roles = Role::pluck('name', 'id');
        return view('employee.create', compact('departments', 'roles'));
    }

    /**
     * Handle registration
     */
    public function register(Request $request)
    {
        $request->validate([
            'username' => 'required|string|max:255|unique:users',
            'password' => 'required|string|min:6|confirmed',
            'department_id' => 'required|exists:departments,id',
            'role_id' => 'required|exists:roles,id',
        ]);

        $user = new User();
        $user->username = $request->username;
        $user->password = Hash::make($request->password);
        $user->department_id = $request->department_id;
        $user->emp_type = 3; // 3=user, 2=admin
        $user->save();

        // Assign role via Spatie
        $role = Role::findById($request->role_id);
        $user->assignRole($role);

        // Create profile
        $profile = new Profile();
        $profile->user()->associate($user);
        $profile->employee_code = 100;
        $profile->save();

        return redirect()->back()->with('success', 'Employee created successfully.');
    }
}
